<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Dtos\Post;

use Erlage\Photogram\Data\Dtos\AbstractDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintValidatorDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintTransformerDTO;

final class PostDisplayUserTagsOnItemDTO extends AbstractDTO
{
    /**
     * to comply with DTO interface
     */
    use TraitComplaintValidatorDTO;
    use TraitComplaintTransformerDTO;

    /**
     * @var PostDisplayUserTagDTO[] 
     */
    private $items = array();

    private $iteratorKey = 0;

    /*
    |--------------------------------------------------------------------------
    | constructor without promoted properties, helps static analysis
    |--------------------------------------------------------------------------
    */

    /**
     * @param PostDisplayUserTagDTO[] $items 
     */
    public function __construct($items = array())
    {
        $this -> items = $items;
    }

    /*
    |--------------------------------------------------------------------------
    | from json map
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public static function fromJsonMap(array $data)
    {
        /**
         * @param PostDisplayUserTagDTO[] $items 
         */
        $items = array();

        foreach ($data as $dataKey => $dataItem)
        {
            $items[] = PostDisplayUserTagDTO::fromJsonMap($dataItem);
        }

        return new static($items);
    }

    /*
    |--------------------------------------------------------------------------
    | interface
    |--------------------------------------------------------------------------
    */

    /**
     * whether post content contains any item
     * method simply runs count() call on items property and checks whether
     * count returns a non-negative number
     */
    public function isEmpty(): bool
    {
        return $this -> getLength() > 0;
    }

    /**
     * get first item from content.
     */
    public function getFirstItem(): PostDisplayUserTagDTO
    {
        return $this -> items[0];
    }

    public function getLength(): int
    {
        return \count($this -> items);
    }

    /**
     * add an item to post content.
     * note: content once added cannot be removed for now. we'll introduce
     * new API for this wrapper that'll include selective selects and deletes
     */
    public function add(PostDisplayUserTagDTO $postDisplayUserTagItemDTO): self
    {
        $this -> items[] = $postDisplayUserTagItemDTO;

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | iterator related
    |--------------------------------------------------------------------------
    */

    public function iteratorPrepare(): void
    {
        $this -> iteratorKey = -1;
    }

    public function iteratorForward(): bool
    {
        if (isset($this -> items[$this -> iteratorKey + 1]))
        {
            $this -> iteratorKey++;

            return true;
        }

        return false;
    }

    public function iteratorCurrentDisplayUserTag(): PostDisplayUserTagDTO
    {
        return $this -> items[$this -> iteratorKey];
    }

    /*
    |--------------------------------------------------------------------------
    | serialization for json_encode
    |--------------------------------------------------------------------------
    */

    public function jsonSerialize()
    {
        return $this -> items;
    }

    /*
    |--------------------------------------------------------------------------
    | implement DTO methods
    |--------------------------------------------------------------------------
    */

    /**
     * @param PostDisplayUserTagsOnItemDTO $postDisplayUserTagItemDTO
     */
    public static function serialize($postDisplayUserTagItemDTO): string
    {
        return \json_encode($postDisplayUserTagItemDTO);
    }

    /**
     * @return PostDisplayUserTagsOnItemDTO
     */
    public static function deSerialize(string $postDisplayUserTagItemDTO)
    {
        $data = \json_decode($postDisplayUserTagItemDTO, true);

        return self::fromJsonMap($data);
    }
}
